<?php
session_start();

// ✅ Load Google reCAPTCHA keys from `config.php`
$config = include('config.php');
$site_key = htmlspecialchars($config['recaptcha']['site-key'], ENT_QUOTES, 'UTF-8');
$secret_key = $config['recaptcha']['secret-key']; // Kept hidden, used only for verification

// ✅ STEP 1: Check if user is verified within the last 10 minutes
$verification_lifetime = 1200; // 10 minutes (600 seconds)
if (!empty($_SESSION['user_verified']) && time() - $_SESSION['captcha_verified_time'] < $verification_lifetime) {
    return; // ✅ Allow access without reCAPTCHA
}

// ✅ STEP 2: If form submitted, verify Google reCAPTCHA
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['g-recaptcha-response'])) {
    $recaptcha_response = $_POST['g-recaptcha-response'];

    // Verify reCAPTCHA with Google
    $verify_url = "https://www.google.com/recaptcha/api/siteverify?secret={$secret_key}&response={$recaptcha_response}";
    $verify_response = json_decode(file_get_contents($verify_url), true);

    // ✅ If reCAPTCHA is successful, mark user as verified
    if ($verify_response["success"]) {
        $_SESSION['user_verified'] = true;
        $_SESSION['captcha_verified_time'] = time(); // Store verification timestamp
        setcookie('human_check', '1', time() + 1200, "/"); // Store verification for 10 minutes
        header("Refresh:0"); // Reload page to continue
        exit;
    } else {
        blockUser();
    }
}

// ✅ STEP 3: Start strict 20-second verification timer
if (empty($_SESSION['captcha_timer'])) {
    $_SESSION['captcha_timer'] = time();
}

// ❌ If more than 20 seconds have passed, block the user
if (time() - $_SESSION['captcha_timer'] > 600) {
    blockUser();
}

// ✅ STEP 4: Display reCAPTCHA verification page
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verify You Are Human</title>
    <script src="https://www.google.com/recaptcha/api.js" async defer></script>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
</head>
<body class="flex items-center justify-center min-h-screen bg-gray-100">

<div class="bg-white p-8 rounded-lg shadow-lg text-center max-w-md">
    <h2 class="text-2xl font-bold text-gray-700 mb-4">Verify You Are Human</h2>
    <p class="text-gray-600">Click on the CAPTCHA to proceed.</p>

    <!-- Loader Animation -->
    <div id="loader" class="flex justify-center mt-4">
        <div class="w-12 h-12 border-4 border-blue-500 border-dotted rounded-full animate-spin"></div>
    </div>

    <!-- CAPTCHA Container (Hidden initially) -->
    <div id="captcha-container" class="hidden mt-4">
        <form id="captchaForm" method="POST">
            <div class="g-recaptcha" data-sitekey="<?= $site_key ?>" data-callback="onCaptchaSuccess"></div>
        </form>
    </div>
</div>

<script>
    // ✅ Show CAPTCHA after a short delay
    setTimeout(() => {
        document.getElementById('loader').classList.add('hidden');
        document.getElementById('captcha-container').classList.remove('hidden');
    }, 2000); // Show after 2 seconds

    // ✅ Auto-submit form on successful CAPTCHA verification
    function onCaptchaSuccess() {
        document.getElementById('captchaForm').submit();
    }

    // ❌ Auto-block user if they fail within 20 seconds
    setTimeout(() => {
        window.location.href = 'login?block=1';
    }, 900000);
</script>

</body>
</html>

<?php
exit;

// 🚫 Function to block the user with a 404 Not Found
function blockUser() {
    header("HTTP/1.1 404 Not Found");
    exit("<h1>404 Not Found</h1><p>The requested page could not be found.</p>");
}
?>
